﻿using System;
using UnityEngine;
using Random = UnityEngine.Random;

//CarAudio handles all engine/car sounds - based on default Unity script
public class CarAudio : MonoBehaviour
{
    public AudioClip GearShiftClip; //Audio clip used for gear shifting
    public AudioClip EngineClip;    //Audio clip for high acceleration
    public float PitchMultiplier = 1.0f;    //Used for altering the pitch of audio clips
    public float LowPitchMinimum = 1.0f;    //The lowest possible pitch for the low sounds
    public float LowPitchMaximum = 6.0f;    //The highest possible pitch for the low sounds
    public float HighPitchMultiplier = 0.25f;   //Used for altering the pitch of high sounds
    public float DopplerLevel = 1;  //The mount of doppler effect used in the audio
    public bool ShouldUseDoppler = true;    //Toggle for using doppler

    private AudioSource _HighAccelSource; //Source for the high acceleration sounds
    private bool _HaveStartedSound; //Flag for knowing if we have started sounds
    private CarController _CarController; //Reference to car we are controlling

    /// <summary>
    /// Starts the engine sound
    /// </summary>
    private void StartSound()
    {
        _CarController = GetComponent<CarController>();
        _HighAccelSource = SetUpEngineAudioSource(EngineClip);
        _HaveStartedSound = true;
    }

    /// <summary>
    /// Stops the engine sound
    /// </summary>
    private void StopSound()
    {
        foreach (var source in GetComponents<AudioSource>())
        {
            Destroy(source);
        }

        _HaveStartedSound = false;
    }

    private void Start()
    {
        GameManager.Instance.GameStateChanged += OnGameStateChanged;
    }

    private void OnGameStateChanged(object sender, GameManager.GameStateChangedEventArgs e)
    {
        //We need to handle the audio depending on the game state
        if (_HighAccelSource != null)
        {
            if (e.NewState != Constants.GameState.Playing)
            {
                if (_HighAccelSource.isPlaying)
                {
                    _HighAccelSource.Pause();
                }
            }

            else
            {
                _HighAccelSource.UnPause();
            }
        }
    }

    private void Update()
    {
        if (!_HaveStartedSound)
        {
            StartSound();
        }

        if (_HaveStartedSound)
        {
            //The pitch is interpolated between the min and max values, according to the car's revs
            float pitch = ULerp(LowPitchMinimum, LowPitchMaximum, _CarController.CurrentRPM);

            //Clamp to minimum pitch (note, not clamped to max for high revs while burning out)
            pitch = Mathf.Min(LowPitchMaximum, pitch);

            _HighAccelSource.pitch = pitch * PitchMultiplier * HighPitchMultiplier;
            _HighAccelSource.dopplerLevel = ShouldUseDoppler ? DopplerLevel : 0;
            _HighAccelSource.volume = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]);
        }
    }

    /// <summary>
    /// Sets up the engine source based on the AudioClip
    /// </summary>
    /// <param name="clip">The audio clip for the engine sound</param>
    /// <returns>A source for the engine sound</returns>
    private AudioSource SetUpEngineAudioSource(AudioClip clip)
    {
        //Create the new audio source component on the game object and set up its properties
        AudioSource source = gameObject.AddComponent<AudioSource>();
        source.clip = clip;
        source.volume = 0;
        source.loop = true;

        //Start the clip from a random point
        source.time = Random.Range(0f, clip.length);
        source.Play();
        source.dopplerLevel = 0;
        return source;
    }

    //Unclamped versions of Lerp and Inverse Lerp, to allow value to exceed the from-to range
    private static float ULerp(float from, float to, float value)
    {
        return (1.0f - value) * from + value * to;
    }
}